%% Example 1 
% Simulation reproducing Figure 4b of 
% Chem. Commun., 2016,52, 2091-2094
clear all
% some constants
rcm = 29979.2458;   %rcm    to MHz
meV = rcm*8.065;    %meV    to MHz
 
% INS specific inputs
Exp.SpectrumType = 'SE';
Exp.Temperature = 30;
Exp.Energy = 0:0.05:11; 
Exp.lwfwhm = 0.87;
Exp.Q = 1.15:0.05:1.65;
Sys.FormFactor = 'Tb3';
Sys.Coords = [0 0 0];
 
% Tb3+ J=6
Sys.S = 6;
 
%Tb-b compound crystal field parameters
Sys.B2 = [0 0 0.712 0 0.0729].*meV;
Sys.B4 = [0 0 0 0 0 0 0 0 0].*meV;
 
% run simulation and plot 
[cross_sec,Eigsb] = mint(Sys,Exp);
figure
plot(Exp.Energy,cross_sec.*3); hold on
 
%Tb-a compound crystal field parameters
Sys.B2 = [0 0 0.705 0 0.0250].*meV;
Sys.B4 = [0 0 0 0 0 0 0 0 -8.5E-4].*meV;
 
% run simulation and plot over previous simulation in a 3 to 7 scaled ratio to match figure 4b of Chem. Commun., 2016,52, 2091-2094
[cross_sec,Eigsa] = mint(Sys,Exp);
plot(Exp.Energy,cross_sec.*7); 
 
% format figure axes
set(gca,'xtick',[0:1:11])
xlabel('Energy (meV)');
ylabel('Intensity (arb. units)');

%% Example 2
% PHYSICAL REVIEW B 88, 064405 (2013)
clear all
% some constants
rcm = 29979.2458;    % reciprocal cm to MHz
meV = rcm*8.065;     % meV           to MHz

% All parameters come from PHYSICAL REVIEW B 88, 064405 (2013)

%Setup of J=6 for Tb3+ and S=1/2 for Cu
Sys.S = [6 1/2];

% Exchange coupling
Jxx = -0.54;
Jyy = -0.54;
Jzz = -0.298;
Sys.ee = [Jxx Jyy Jzz].*meV;
Sys.Coords = [0 0 0; 3.48 0 0]; % distance between Tb and Cu

%hyperfine term 
Sys.Nucs = '159Tb,63Cu';
Sys.A = [6.2e-3 6.2e-3 6.2e-3 0 0 0 ; 0 0 0 0.043 0.043 0.043].*meV;

Sys.FormFactor = {'Tb3','Cu2'}; %elements included
Sys.Coords = [0 0 0; 3.48 0 0]; % distance between Tb and Cu

%Stevens parameters
B20 = -0.197;
B22 = 0.066;
B40 = 1.5e-4;
B44 = 5e-4;
B60 = -1.25e-5;
B64 = 1.17e-4;
Sys.B2 = [B22 0 B20 0 0 ; zeros(1,5)].*meV;        % B(k=2,q) with q = +2,+1,0,-1,-2
Sys.B4 = [B44 0 0 0 B40 0 0 0 0 ; zeros(1,9)].*meV;  % B(k=4,q) with q = +4,+3,+2,+1,0,-1,-2,-3,-4
Sys.B6 = [0 0 B64 0 0 0 B60 0 0 0 0 0 0 ; zeros(1,13)].*meV;  % B(k=6,q) with q = +4,+3,+2,+1,0,-1,-2,-3,-4

% SE calculation
Exp.Energy = [0:0.01:16]; % spectrum energy range in meV
Exp.lwfwhm = .3; % gaussin broadening FWHM in meV
Exp.Q = [0.1,0.5,1,1.5,2]; % Q range in A^-1
Exp.SpectrumType = 'SE'; % S(E) only
Exp.Temperature = 3.5; % Kelvin

[cross_sect,Eigs]=mint(Sys,Exp);

% SE calculation
Exp.Energy = [1.2:0.001:2.2]; % spectrum energy range in meV
Exp.lwfwhm = .03; % gaussin broadening FWHM in meV
Exp.Q = [0.3:0.1:1]; % Q range in A^-1
Exp.SpectrumType = 'SE'; % S(E) only
Exp.Temperature = 3.5; % Kelvin
[cross_sect,Eigs]=mint(Sys,Exp);

%make figure 5a
figure
plot(Exp.Energy,cross_sect); hold on
xlabel('Energy (meV)')
ylabel('Intensity (arb. units)')
set(gca,'xtick',[1.2:0.2:2.2])
axis tight

% SQE map calculation, simulation of figure 2
Exp.Energy = [0:0.05:15]; % spectrum energy range in meV
Exp.Q = [0.1:0.05:3]; %Q range in A^-1
Exp.lwfwhm = .3; % gaussin broadening FWHM in meV
Exp.SpectrumType = 'SQE' ; % SQE map calculation
[cross_sect,Eigs]=mint(Sys,Exp);

%make figure
figure
surf(Exp.Q,Exp.Energy,cross_sect)
view(0,90); shading interp; colorbar; axis tight
caxis([0,4]);
xlabel('Q ($\AA^{-1}$)','interpreter','latex')
ylabel('Energy (meV)')

%make SQ plot
Exp.SpectrumType = 'SQ';
Exp.Q = [0:0.05:5]; %Q range in A^-1
Exp.InitialStates = [1:32]; % ground doublet
Exp.FinalStates = [33:64]; % to exchange split excited states
[cross_sect1,Eigs]=mint(Sys,Exp);

Exp.InitialStates = [1:32]; % ground doublet
Exp.FinalStates = [65:160]; % to CF and exchange split excited states
[cross_sect2,Eigs]=mint(Sys,Exp);

figure
plot(Exp.Q,cross_sect1,Exp.Q,cross_sect2); hold on
xlabel('Q ($\AA^{-1}$)','interpreter','latex')
ylabel('Intensity (arb. units)')
legend('peak i','peak ii')

%% Example 3
% Reproduces Fig 6 of Dalton Trans., 2011, 40, 2725-2734.
clear all;
% some constants
rcm = 29979.2458;   %rcm    to MHz
meV = rcm*8.065;    %meV    to MHz
 
% setup of the Hamiltonian in the usual easyspin way
D = -0.041.*meV; % convert from meV to MHz
E = 0.007.*meV; % convert from meV to MHz
 

Sys.S = [3/2 3/2 3/2 3/2 3/2 3/2];
Sys.D = [D E;D E;D E;D E;D E;D E];
Sys.J = [1.46 0 0 0 0 1.46 0 0 0 1.46 0 0 1.46 0 1.46].*meV; % convert from meV to MHz
Sys.FormFactor = 'Cr3';
% Cr ion coords in angstroms
Sys.Coords =   [      0         0         0 ;
                -3.3807    0.0840   -0.2765 ;
                -5.7290   -2.3170    0.1103 ;
                -5.5960   -5.6470   -0.2956 ;
                -3.1148   -7.8270    0.2920 ;
                 0.2482   -7.7410   -0.1803]; 

Opt.NumEigs = 100;

Exp.SpectrumType = 'SE';
Exp.Temperature = [1.5, 6, 15];
Exp.Energy = -1:0.04:4;
Exp.lwfwhm = 0.25;
Exp.Q = [0.1,0.5,1,1.5,2];
 [cross,Eigs] = mint(Sys,Exp,Opt);
figure
plot(Exp.Energy,cross)
xlim([0 3.25])
xlabel('Energy (meV)')
ylabel('Intensity (arb. units)')
legend('1.5 K','6 K','15 K')

 

%make Fig S14 of Dalton Trans., 2011, 40, 2725-2734.
Exp.SpectrumType = 'SQ';
Exp.Q = [0:0.05:5]; %Q range in A^-1
Exp.InitialStates = 2:4; 
Exp.FinalStates = 5:9; 
Exp.Temperature = 6;
[cross_sectQ,Eigs]=mint(Sys,Exp,Opt);

figure
plot(Exp.Q,cross_sectQ); hold on
xlabel('Q ($\AA^{-1}$)','interpreter','latex')
ylabel('Intensity (arb. units)')
xlim([0 3.25])

%% Example 4

% reproducing Nat Phys, 2012, 8, 906 Figure 3 parts d e and f
clear all;
% some constants
rcm = 29979.2458;   %rcm    to MHz
meV = rcm*8.065;    %meV    to MHz
 
% setup of the Hamiltonian in the usual easyspin way
D = -0.041.*meV; % convert from meV to MHz
E = 0.007.*meV; % convert from meV to MHz
 
Sys.S = [3/2 3/2 3/2 3/2 3/2 3/2 3/2 3/2];
Sys.D = [D E;D E;D E;D E;D E;D E;D E;D E];
Sys.J = [1.46 0 0 0 0 0 1.46 1.46 0 0 0 0 0 1.46 0 0 0 0 1.46 0 0 0 1.46 0 0 1.46 0 1.47].*meV; % convert from meV to MHz
Sys.FormFactor = 'Cr3';
 
% include eigenvalues up to 6 meV only
Opt.NumEigs = 55; 
 
Exp.SpectrumType = 'SQxyz';
Exp.Temperature = 1.5;
 
% coordinates for molecule 1 in the unit cell
Sys.Coords =    [0.000 0.000  0.000;
2.971 1.621 -0.154;
3.947 4.860  0.000;
2.325 7.832 -0.154;
-0.914 8.807  0.000;
-3.885 7.186 -0.154;
-4.861 3.947  0.000;
-3.239 0.975 -0.154];
 
% figure 3d molecule 1
Exp.InitialStates = 1;
Exp.FinalStates = 2:4;
Exp.Qx = -2:0.1:2; Exp.Qy = -2:0.1:2; Exp.Qz = 0;
[cross1_mol1,Eigs] = mint(Sys,Exp,Opt);
 
% figure 3e molecule 1
Exp.InitialStates = 1;
Exp.FinalStates = 10:21;
Exp.Qx = -3:0.1:3; Exp.Qy = -3:0.1:3; Exp.Qz = 0;
[cross2_mol1,Eigs] = mint(Sys,Exp,Opt);
 
% figure 3f molecule 1
Exp.InitialStates = 1;
Exp.FinalStates = 29:55;
Exp.Qx = -2:0.1:2; Exp.Qy = -2:0.1:2; Exp.Qz = 0;
[cross3_mol1,Eigs] = mint(Sys,Exp,Opt);
 
% coordinates for molecule 2 in the unit cell
Sys.Coords =    [0.000 0.000  0.000
3.239 0.976 0.154;
4.860 3.947 0.000;
3.885 7.186 0.154;
0.913 8.808 0.000;
-2.326 7.832 0.154;
-3.947 4.861 0.000;
-2.972 1.622 0.154];
 
% figure 3d molecule 2
Exp.InitialStates = 1;
Exp.FinalStates = 2:4;
Exp.Qx = -2:0.1:2; Exp.Qy = -2:0.1:2; Exp.Qz = 0;
[cross1_mol2,Eigs] = mint(Sys,Exp,Opt);
 
% figure 3e molecule 2
Exp.InitialStates = 1;
Exp.FinalStates = 10:21;
Exp.Qx = -3:0.1:3; Exp.Qy = -3:0.1:3; Exp.Qz = 0;
[cross2_mol2,Eigs] = mint(Sys,Exp,Opt);
 
% figure 3f molecule 2
Exp.InitialStates = 1;
Exp.FinalStates = 29:55;
Exp.Qx = -2:0.1:2; Exp.Qy = -2:0.1:2; Exp.Qz = 0;
[cross3_mol2,Eigs] = mint(Sys,Exp,Opt);
 
figure
subplot(1,3,1)
plotSQxyz(-2:0.1:2,-2:0.1:2,0,cross1_mol1+cross1_mol2); hold on; caxis([0 18]);
xlabel('Qx ($\AA^{-1}$)','interpreter','latex')
ylabel('Qy ($\AA^{-1}$)','interpreter','latex')
view(0,90)
caxis([0 70]);
colormap jet
caxis
axis tight
subplot(1,3,2)
plotSQxyz(-3:0.1:3,-3:0.1:3,0,cross2_mol1+cross2_mol2); hold on; caxis([0 18]);
xlabel('Qx ($\AA^{-1}$)','interpreter','latex')
ylabel('Qy ($\AA^{-1}$)','interpreter','latex')
view(0,90)
caxis([0 18]);
colormap jet
caxis
axis tight
subplot(1,3,3)
plotSQxyz(-2:0.1:2,-2:0.1:2,0,cross3_mol1+cross3_mol2); hold on; caxis([0 18]);
xlabel('Qx ($\AA^{-1}$)','interpreter','latex')
ylabel('Qy ($\AA^{-1}$)','interpreter','latex')
view(0,90)
caxis([0 12.85]);
colormap jet
caxis
axis tight



% make a 3D slice simulating Figure 2 from Nat Phys, 2012, 8, 906
Exp.InitialStates = 1;
Exp.FinalStates = 2:4;
Opt.NumEigs = 4; 
 
% coordinates for molecule 1 in the unit cell
Sys.Coords =    [0.000 0.000  0.000;
2.971 1.621 -0.154;
3.947 4.860  0.000;
2.325 7.832 -0.154;
-0.914 8.807  0.000;
-3.885 7.186 -0.154;
-4.861 3.947  0.000;
-3.239 0.975 -0.154];
 
Exp.Qx = -2:0.1:2; Exp.Qy = -2:0.1:2; Exp.Qz = 0;
[cross4_mol1,Eigs] = mint(Sys,Exp,Opt);
 
Exp.Qx = -2:0.1:2; Exp.Qy = 0; Exp.Qz = -0.5:0.05:0.5;
[cross5_mol1,Eigs] = mint(Sys,Exp,Opt);
 
Exp.Qx = 0; Exp.Qy = -2:0.1:2; Exp.Qz = -0.5:0.05:0.5;
[cross6_mol1,Eigs] = mint(Sys,Exp,Opt);
 
% coordinates for molecule 2 in the unit cell
Sys.Coords =    [0.000 0.000  0.000
3.239 0.976 0.154;
4.860 3.947 0.000;
3.885 7.186 0.154;
0.913 8.808 0.000;
-2.326 7.832 0.154;
-3.947 4.861 0.000;
-2.972 1.622 0.154];
 
Exp.Qx = -2:0.1:2; Exp.Qy = -2:0.1:2; Exp.Qz = 0;
[cross4_mol2,Eigs] = mint(Sys,Exp,Opt);
 
Exp.Qx = -2:0.1:2; Exp.Qy = 0; Exp.Qz = -0.5:0.05:0.5;
[cross5_mol2,Eigs] = mint(Sys,Exp,Opt);
 
Exp.Qx = 0; Exp.Qy = -2:0.1:2; Exp.Qz = -0.5:0.05:0.5;
[cross6_mol2,Eigs] = mint(Sys,Exp,Opt);
 
figure
plotSQxyz(-2:0.1:2,-2:0.1:2,0,cross4_mol1+cross4_mol2); hold on; caxis([0 80]);
plotSQxyz(-2:0.1:2,0,-0.5:0.05:0.5,cross5_mol1+cross5_mol2); hold on; caxis([0 80]);
plotSQxyz(0,-2:0.1:2,-0.5:0.05:0.5,cross6_mol1+cross6_mol2); hold on; caxis([0 80]);
xlabel('Qx ($\AA^{-1}$)','interpreter','latex')
ylabel('Qy ($\AA^{-1}$)','interpreter','latex')
colormap jet
caxis
axis equal
